//	Altirra - Atari 800/800XL emulator
//	Copyright (C) 2008 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef AT_GTIA_H
#define AT_GTIA_H

#include <vd2/system/refcount.h>
#include <vd2/system/vdstl.h>
#include <vd2/Kasumi/pixmap.h>

class IVDVideoDisplay;
class VDVideoDisplayFrame;

class IATGTIAEmulatorConnections {
public:
	virtual uint32 GTIAGetXClock() = 0;
	virtual void GTIASetSpeaker(bool state) = 0;
	virtual void GTIARequestAnticSync() = 0;
};

class ATFrameTracker;

class ATGTIAEmulator {
public:
	ATGTIAEmulator();
	~ATGTIAEmulator();

	void Init(IATGTIAEmulatorConnections *);
	
	enum AnalysisMode {
		kAnalyzeNone,
		kAnalyzeColors,
		kAnalyzeLayers,
		kAnalyzeDList,
		kAnalyzePM,
		kAnalyzeCount
	};

	void AdjustColors(double baseDelta, double rangeDelta);

	AnalysisMode GetAnalysisMode() const { return mAnalysisMode; }
	void SetAnalysisMode(AnalysisMode mode);

	void SetFrameSkip(bool turbo) { mbTurbo = turbo; }
	void SetVideoOutput(IVDVideoDisplay *pDisplay);
	void SetStatusFlags(uint32 flags) { mStatusFlags |= flags; mStickyStatusFlags |= flags; }
	void ResetStatusFlags(uint32 flags) { mStatusFlags &= ~flags; }
	void PulseStatusFlags(uint32 flags) { mStickyStatusFlags |= flags; }

	void SetCassetteIndicatorVisible(bool vis) { mbShowCassetteIndicator = vis; }
	void SetCassettePosition(float pos);

	void SetPALMode(bool enabled);

	void SetConsoleSwitch(uint8 c, bool down);
	void SetControllerTrigger(int index, bool state) { mTRIG[index] = state ? 0x00 : 0x01; }

	void DumpStatus();

	void BeginScanline(int y, bool hires);
	void EndScanline(uint8 dlControl);
	void UpdatePlayer(int index, uint8 byte);
	void UpdateMissile(uint8 byte);
	void UpdatePlayfield160(uint32 x, uint8 byte);
	void UpdatePlayfield320(uint32 x, uint8 byte);
	void EndPlayfield();
	void Sync();

	void RenderActivityMap(const uint8 *src);
	void UpdateScreen();
	void RecomputePalette();

	uint8 ReadByte(uint8 reg);
	void WriteByte(uint8 reg, uint8 value);

protected:
	void ApplyArtifacting();

	IATGTIAEmulatorConnections *mpConn; 
	IVDVideoDisplay *mpDisplay;
	uint32	mX;
	uint32	mY;
	uint32	mLastSyncX;

	sint32	mPlayerShifter0;
	sint32	mPlayerShifter1;
	sint32	mPlayerShifter2;
	sint32	mPlayerShifter3;
	sint32	mMissileShifter0;
	sint32	mMissileShifter1;
	sint32	mMissileShifter2;
	sint32	mMissileShifter3;

	AnalysisMode	mAnalysisMode;
	uint8	mArtifactMode;

	uint8	mPlayerPos[4];
	uint8	mMissilePos[4];
	uint8	mPlayerSize[4];
	uint8	mPlayerData[4];
	uint8	mDelayedPlayerData[4];
	uint8	mMissileData;
	uint8	mDelayedMissileData;
	uint8	mMissileSize;
	uint8	mPMColor[4];		// $D012-D015 player and missile colors
	uint8	mPFColor[4];		// $D016-D019 playfield colors
	uint8	mPFBAK;				// $D01A background color
	uint8	mPRIOR;				// $D01B priority
	uint8	mVDELAY;			// $D01C vertical delay
	uint8	mGRACTL;			// $D01D
								// bit 2: latch trigger inputs
								// bit 1: enable players
								// bit 0: enable missiles
	uint8	mCONSOL;			// $D01F
								// bit 3: speaker output

	uint8	mTRIG[4];

	uint8	mPlayerCollFlags[4];
	uint8	mMissileCollFlags[4];

	uint8	*mpDst;
	vdrefptr<VDVideoDisplayFrame>	mpFrame;
	ATFrameTracker *mpFrameTracker;
	bool	mbHiresMode;
	bool	mbTurbo;
	bool	mbPALMode;
	bool	mbShowCassetteIndicator;

	uint32	mStatusFlags;
	uint32	mStickyStatusFlags;
	float	mCassettePos;

	const uint8 *mpPriTable;
	const uint8 *mpMissileTable;

	uint8	mColorTable[19];
	uint8	mMergeBuffer[456];
	uint8	mPlayfieldLumaBytes[57+1];
	uint8	mPriorityTables[32][256];
	uint32	mPalette[256];

	double	mPaletteColorBase;
	double	mPaletteColorRange;

	vdfastvector<uint8>		mPreArtifactFrameBuffer;
	VDPixmap	mPreArtifactFrame;
};

#endif
